<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Loan extends Model
{
    protected $fillable = [
        'user_id',
        'amount',
        'purpose',
        'status',
        'leader_approved_by',
        'leader_approved_at',
        'approved_by',
        'approved_at',
        'payment_date',
        'payment_amount',
        'payment_method',
        'notes',
        'disbursed_by',
        'disbursed_at',
    ];

    protected $casts = [
        'leader_approved_at' => 'datetime',
        'approved_at' => 'datetime',
        'payment_date' => 'datetime',
        'disbursed_at' => 'datetime',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function leaderApprover()
    {
        return $this->belongsTo(User::class, 'leader_approved_by');
    }

    public function approver()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function disburser()
    {
        return $this->belongsTo(User::class, 'disbursed_by');
    }

    // Helper methods for two-tier approval statuses
    public function isPendingLeaderReview()
    {
        return $this->status === 'pending_leader_review';
    }

    public function isPendingTopLeaderReview()
    {
        return $this->status === 'pending_top_leader_review';
    }

    public function isLeaderRejected()
    {
        return $this->status === 'leader_rejected';
    }

    public function isApproved()
    {
        return $this->status === 'approved';
    }

    public function isTopLeaderRejected()
    {
        return $this->status === 'top_leader_rejected';
    }

    public function isRejected()
    {
        return $this->isLeaderRejected() || $this->isTopLeaderRejected();
    }

    // Legacy method kept for backward compatibility
    public function isPending()
    {
        return $this->isPendingLeaderReview() || $this->isPendingTopLeaderReview();
    }

    public function isPaid()
    {
        return $this->disbursed_at !== null;
    }

    public function getStatusBadgeColor()
    {
        return match($this->status) {
            'pending_leader_review' => 'bg-yellow-100 text-yellow-800',
            'pending_top_leader_review' => 'bg-orange-100 text-orange-800',
            'leader_rejected' => 'bg-red-100 text-red-800',
            'approved' => $this->isPaid() ? 'bg-blue-100 text-blue-800' : 'bg-green-100 text-green-800',
            'top_leader_rejected' => 'bg-red-100 text-red-800',
            default => 'bg-gray-100 text-gray-800',
        };
    }

    public function getStatusText()
    {
        if ($this->isPaid()) {
            return 'Disbursed';
        }
        
        return match($this->status) {
            'pending_leader_review' => 'Pending Leader Review',
            'pending_top_leader_review' => 'Pending Final Approval',
            'leader_rejected' => 'Rejected by Leader',
            'approved' => 'Approved',
            'top_leader_rejected' => 'Rejected by Top-Leader',
            default => ucfirst($this->status),
        };
    }
}
